# ##### BEGIN GPL LICENSE BLOCK #####
#
#  keKit Deprecated - Deprecated items from keKit (Tool Kit / Script Collection for Blender)
#  Copyright (C) 2024  Kjell Emanuelsson (ke-code.xyz)
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see https://www.gnu.org/licenses
#
# ##### END GPL LICENSE BLOCK #####
import bpy
import rna_keymap_ui
from bpy.types import Panel, Operator, AddonPreferences, Menu
from mathutils import Vector
from bpy.props import BoolProperty, StringProperty, FloatProperty
from bpy_extras.view3d_utils import region_2d_to_vector_3d, region_2d_to_origin_3d

bl_info = {
    "name": "keKit Deprecated",
    "author": "Kjell Emanuelsson",
    "category": "",
    "version": (1, 0, 5),
    "blender": (2, 80, 0),
    "location": "N-Panel/Custom",
    "warning": "",
    "description": "Discontinued items from keKit",
    "doc_url": "ke-code.xyz",
}


def get_prefs():
    return bpy.context.preferences.addons['kekit'].preferences


#
# UI PANELS
#
class UIModesModule(Panel):
    bl_idname = "UI_PT_M_MODES"
    bl_label = "Modes"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'kekit'

    def draw(self, context):
        prefs = context.preferences.addons[__name__].preferences
        layout = self.layout
        row = layout.row(align=False)
        row.alignment = "CENTER"
        row.operator('ked.selmode', text="", icon="VERTEXSEL").edit_mode = "VERT"
        row.operator('ked.selmode', text="", icon="EDGESEL").edit_mode = "EDGE"
        row.operator('ked.selmode', text="", icon="FACESEL").edit_mode = "FACE"
        row.operator('ked.selmode', text="", icon="OBJECT_DATAMODE").edit_mode = "OBJECT"
        row.operator('view3d.object_mode_pie_or_toggle', text="", icon="THREE_DOTS")
        row.prop(prefs, "selmode_mouse", text="")
        row.operator('ked.spacetoggle', text="", icon="MOUSE_MOVE")
        if prefs.show_misc:
            layout.operator('ked.mod_em_vis')


#
# PIE MENUS
#
class KedPieFitPrimAdd(Menu):
    bl_idname = "VIEW3D_MT_ked_pie_fitprim_add"
    bl_label = "FitPrim+Add"

    @classmethod
    def poll(cls, context):
        return context.space_data.type == "VIEW_3D" and is_registered("VIEW3D_OT_ke_fitprim")

    def draw(self, context):
        cm = context.mode
        layout = self.layout
        pie = layout.menu_pie()

        if cm == "EDIT_MESH":

            w = pie.operator("view3d.ke_fitprim", text="Cylinder", icon='MESH_CYLINDER')
            w.ke_fitprim_option = "CYL"
            w.ke_fitprim_pieslot = "W"

            e = pie.operator("view3d.ke_fitprim", text="Cylinder Obj", icon='MESH_CYLINDER')
            e.ke_fitprim_option = "CYL"
            e.ke_fitprim_pieslot = "E"
            e.ke_fitprim_itemize = True

            s = pie.operator("view3d.ke_fitprim", text="Cube", icon='CUBE')
            s.ke_fitprim_option = "BOX"
            s.ke_fitprim_pieslot = "S"

            n = pie.operator("view3d.ke_fitprim", text="Cube Obj", icon='MESH_CUBE')
            n.ke_fitprim_option = "BOX"
            n.ke_fitprim_pieslot = "N"
            n.ke_fitprim_itemize = True

            col = pie.box().column()
            nw = col.operator("view3d.ke_fitprim", text="Sphere", icon='SPHERE')
            nw.ke_fitprim_option = "SPHERE"
            nw.ke_fitprim_pieslot = "NW"
            nw2 = col.operator("view3d.ke_fitprim", text="QuadSphere", icon='SPHERE')
            nw2.ke_fitprim_option = "QUADSPHERE"
            nw2.ke_fitprim_pieslot = "NW"

            col = pie.box().column()
            ne = col.operator("view3d.ke_fitprim", text="Sphere Obj", icon='MESH_UVSPHERE')
            ne.ke_fitprim_option = "SPHERE"
            ne.ke_fitprim_pieslot = "NE"
            ne.ke_fitprim_itemize = True
            ne2 = col.operator("view3d.ke_fitprim", text="QuadSphere Obj", icon='MESH_UVSPHERE')
            ne2.ke_fitprim_option = "QUADSPHERE"
            ne2.ke_fitprim_pieslot = "NE"
            ne2.ke_fitprim_itemize = True

            sw = pie.operator("view3d.ke_fitprim", text="Plane", icon='MESH_PLANE')
            sw.ke_fitprim_option = "PLANE"
            sw.ke_fitprim_pieslot = "SW"

            se = pie.operator("view3d.ke_fitprim", text="Plane Obj", icon='MESH_PLANE')
            se.ke_fitprim_option = "PLANE"
            se.ke_fitprim_pieslot = "SE"
            se.ke_fitprim_itemize = True

        if cm == "OBJECT":
            # WEST
            op = pie.operator("view3d.ke_fitprim", text="Cylinder", icon='MESH_CYLINDER')
            op.ke_fitprim_option = "CYL"
            op.ke_fitprim_pieslot = "W"

            # EAST
            c = pie.row()
            s = c.column()
            s.separator(factor=27)
            box = s.box()
            box.emboss = "PULLDOWN_MENU"
            col = box.grid_flow(columns=2, even_columns=True, align=True)
            col.scale_x = 1.5
            col.scale_y = 0.9
            col.operator('object.empty_add', icon='EMPTY_AXIS').type = "PLAIN_AXES"
            col.menu_contents("VIEW3D_MT_mesh_add")
            col.menu_contents("VIEW3D_MT_add")

            # SOUTH
            op = pie.operator("view3d.ke_fitprim", text="Plane", icon='MESH_PLANE')
            op.ke_fitprim_option = "PLANE"
            op.ke_fitprim_pieslot = "S"

            # NORTH
            op = pie.operator("view3d.ke_fitprim", text="Cube", icon='MESH_CUBE')
            op.ke_fitprim_option = "BOX"
            op.ke_fitprim_pieslot = "N"

            # NORTHWEST
            op = pie.operator("view3d.ke_fitprim", text="Sphere", icon='MESH_UVSPHERE')
            op.ke_fitprim_option = "SPHERE"
            op.ke_fitprim_pieslot = "NW"

            # NORTHEAST
            pie.separator()

            # SOUTHWEST
            op = pie.operator("view3d.ke_fitprim", text="QuadSphere", icon='MESH_UVSPHERE')
            op.ke_fitprim_option = "QUADSPHERE"
            op.ke_fitprim_pieslot = "SW"
            # SOUTHEAST - BLANK


class KedPieFit2Grid(Menu):
    bl_label = "kedFit2Grid"
    bl_idname = "VIEW3D_MT_ked_pie_fit2grid"

    @classmethod
    def poll(cls, context):
        return context.space_data.type == "VIEW_3D" and is_registered("VIEW3D_OT_ke_fit2grid")

    def draw(self, context):
        layout = self.layout
        pie = layout.menu_pie()
        pie.operator("view3d.ke_fit2grid", text="50cm").set_grid = 0.5
        pie.operator("view3d.ke_fit2grid", text="5cm").set_grid = 0.05
        pie.operator("view3d.ke_fit2grid", text="15cm").set_grid = 0.15
        pie.operator("view3d.ke_fit2grid", text="1cm").set_grid = 0.01
        pie.operator("view3d.ke_fit2grid", text="1m").set_grid = 1.0
        pie.operator("view3d.ke_fit2grid", text="2.5cm").set_grid = 0.025
        pie.operator("view3d.ke_fit2grid", text="25cm").set_grid = 0.25
        pie.operator("view3d.ke_fit2grid", text="10cm").set_grid = 0.1


class KedPieFit2GridMicro(Menu):
    bl_label = "kedFit2Grid_micro"
    bl_idname = "VIEW3D_MT_ked_pie_fit2grid_micro"

    @classmethod
    def poll(cls, context):
        return context.space_data.type == "VIEW_3D" and is_registered("VIEW3D_OT_ke_fit2grid")

    def draw(self, context):
        layout = self.layout
        pie = layout.menu_pie()
        pie.operator("view3d.ke_fit2grid", text="5mm").set_grid = 0.005
        pie.operator("view3d.ke_fit2grid", text="5µm").set_grid = 0.0005
        pie.operator("view3d.ke_fit2grid", text="1.5mm").set_grid = 0.0015
        pie.operator("view3d.ke_fit2grid", text="1µm").set_grid = 0.0001
        pie.operator("view3d.ke_fit2grid", text="1cm").set_grid = 0.01
        pie.operator("view3d.ke_fit2grid", text="2.5µm").set_grid = 0.00025
        pie.operator("view3d.ke_fit2grid", text="5µm").set_grid = 0.0005
        pie.operator("view3d.ke_fit2grid", text="1mm").set_grid = 0.001


class KedPieOrientPivot(Menu):
    bl_label = "kedOrientPivot"
    bl_idname = "VIEW3D_MT_ked_pie_orientpivot"

    @classmethod
    def poll(cls, context):
        return context.space_data.type == "VIEW_3D"

    def draw(self, context):
        mode = bpy.context.mode
        obj = context.active_object
        k = get_prefs()
        ct = context.scene.tool_settings
        name1 = k.opc1_name
        name2 = k.opc2_name
        name3 = k.opc3_name
        name4 = k.opc4_name

        layout = self.layout
        pie = layout.menu_pie()

        c = pie.column()
        cbox = c.box().column(align=True)
        cbox.scale_y = 1.15
        cbox.ui_units_x = 6.25
        cbox.prop(context.scene.transform_orientation_slots[0], "type", expand=True)
        c.separator(factor=13)

        c = pie.column()
        cbox = c.box().column(align=True)
        cbox.scale_y = 1.2
        cbox.ui_units_x = 6.5
        cbox.prop_enum(context.scene.tool_settings, "transform_pivot_point", value='BOUNDING_BOX_CENTER')
        cbox.prop_enum(context.scene.tool_settings, "transform_pivot_point", value='CURSOR')
        cbox.prop_enum(context.scene.tool_settings, "transform_pivot_point", value='INDIVIDUAL_ORIGINS')
        cbox.prop_enum(context.scene.tool_settings, "transform_pivot_point", value='MEDIAN_POINT')
        cbox.prop_enum(context.scene.tool_settings, "transform_pivot_point", value='ACTIVE_ELEMENT')
        if (obj is None) or (mode in {'OBJECT', 'POSE', 'WEIGHT_PAINT'}):
            cbox.prop(context.scene.tool_settings, "use_transform_pivot_point_align")
        else:
            c.separator(factor=2)
        c.separator(factor=14.5)

        pie.operator("view3d.ke_opc", text="%s" % name1, icon="KEYTYPE_JITTER_VEC").combo = "1"
        pie.operator("view3d.ke_opc", text="%s" % name2, icon="KEYTYPE_EXTREME_VEC").combo = "2"
        pie.separator()
        pie.separator()
        pie.operator("view3d.ke_opc", text="%s" % name3, icon="KEYTYPE_MOVING_HOLD_VEC").combo = "3"
        pie.operator("view3d.ke_opc", text="%s" % name4, icon="KEYTYPE_KEYFRAME_VEC").combo = "4"


#
# FUNCTIONS
#
def is_registered(idname_c):
    """Check if class is registered (cstring) E.g: 'VIEW3D_OT_ke_###' """
    try:
        return hasattr(bpy.types, idname_c)
    except AttributeError or ValueError:
        return False


def obj_raycast(obj, matrix, ray_origin, ray_target):
    matrix_inv = matrix.inverted()
    ray_origin_obj = matrix_inv @ ray_origin
    ray_target_obj = matrix_inv @ ray_target
    ray_direction_obj = ray_target_obj - ray_origin_obj
    success, location, normal, face_index = obj.ray_cast(ray_origin_obj, ray_direction_obj)

    if success:
        return location, normal, face_index
    else:
        return None, None, None


def correct_normal(world_matrix, vec_normal):
    n = (world_matrix.to_quaternion() @ vec_normal).to_4d()
    n.w = 0
    return world_matrix.to_quaternion() @ (world_matrix.inverted() @ n).to_3d().normalized()


def mouse_raycast(context, mouse_pos, evaluated=False):
    region = context.region
    rv3d = context.region_data

    view_vector = region_2d_to_vector_3d(region, rv3d, mouse_pos)
    ray_origin = region_2d_to_origin_3d(region, rv3d, mouse_pos)
    ray_target = ray_origin + view_vector

    hit_length_squared = -1.0
    hit_obj, hit_wloc, hit_normal, hit_face = None, None, None, None
    hit, normal, face_index = None, None, None

    cat = ["MESH"]
    # todo: Make hit curves returnable objs w. evaluated?

    if not evaluated:
        objects = [o for o in context.visible_objects]
        depsgraph = objects.copy()
        objects = [o.name for o in context.visible_objects]
    else:
        objects = [o.name for o in context.visible_objects]
        depsgraph = context.evaluated_depsgraph_get()
        depsgraph = depsgraph.object_instances

    for dup in depsgraph:
        if evaluated:
            if dup.is_instance:
                obj = dup.instance_object
                obj_mtx = dup.matrix_world.copy()
            else:
                obj = dup.object
                obj_mtx = obj.matrix_world.copy()
        else:
            obj = dup
            obj_mtx = obj.matrix_world.copy()

        if obj.type in cat and obj.name in objects:
            try:
                hit, normal, face_index = obj_raycast(obj, obj_mtx, ray_origin, ray_target)
            except RuntimeError:
                print("Raycast Failed: Unsupported object type?")
                pass

            if hit is not None:
                hit_world = obj_mtx @ hit
                length_squared = (hit_world - ray_origin).length_squared
                if hit_obj is None or length_squared < hit_length_squared:
                    hit_normal = correct_normal(obj_mtx, normal)
                    hit_wloc = hit_world
                    hit_face = face_index
                    hit_length_squared = length_squared
                    hit_obj = obj

    if hit_obj:
        return hit_obj, hit_wloc, hit_normal, hit_face
    else:
        return None, None, None, None


#
# OPERATORS
#
class KedModEmVis(Operator):
    bl_idname = "ked.mod_em_vis"
    bl_label = "Edit-Mode Visibility"
    bl_description = "Sets ALL modifiers Edit-Mode Visibility to OFF on selected objects\n(or ON,in redo panel)"
    bl_options = {'REGISTER', 'UNDO'}

    mode: BoolProperty(name="Modifiers Visibile", default=False,
                       description="Selected Objects: Sets ALL modifiers Edit Mode visibility to OFF\n"
                                   "(or ON,in redo panel)")

    @classmethod
    def poll(cls, context):
        return context.selected_objects is not None and context.mode == "OBJECT"

    def execute(self, context):
        for obj in context.selected_objects:
            if obj.type == "GPENCIL":
                mods = obj.grease_pencil_modifiers
            else:
                mods = obj.modifiers
            if mods:
                for m in mods:
                    m.show_in_editmode = self.mode
        return {'FINISHED'}


class KedSelectionMode(Operator):
    bl_idname = "ked.selmode"
    bl_label = "Direct Element <-> Object Mode Switch"
    bl_description = "Sets Element Mode - Direct to selection mode, also from Object Mode'\n" \
                     "Legacy Feature - Needed for 'Mouse Over Element Select Mode"

    edit_mode: bpy.props.EnumProperty(
        items=[("VERT", "Vertex Edit Mode", "", 1),
               ("EDGE", "Edge Edit Mode", "", 2),
               ("FACE", "Face Edit Mode", "", 3),
               ("OBJECT", "Object Mode", "", 4)],
        name="Edit Mode",
        default="OBJECT")

    mouse_pos = [0, 0]

    @classmethod
    def description(cls, context, properties):
        if properties.edit_mode == "VERT":
            return "Direct to Vertex Element Mode in Edit Mode'\n" \
                   "Legacy Feature - Needed for 'Mouse Over Element Select Mode"
        elif properties.edit_mode == "EDGE":
            return "Direct to Edge Element Mode in Edit Mode'\n" \
                   "Legacy Feature - Needed for 'Mouse Over Element Select Mode"
        elif properties.edit_mode == "FACE":
            return "Direct to Face Element Mode in Edit Mode'\n" \
                   "Legacy Feature - Needed for 'Mouse Over Element Select Mode"
        elif properties.edit_mode == "OBJECT":
            return "Direct to Object Mode'\n" \
                   "Legacy Feature - Needed for 'Mouse Over Element Select Mode"
        else:
            return "Set Element Mode - Direct to selection mode from Object Mode'\n" \
                   "Legacy Feature - Needed for 'Mouse Over Element Select Mode"

    @classmethod
    def poll(cls, context):
        return context.object is not None

    def invoke(self, context, event):
        self.mouse_pos[0] = event.mouse_region_x
        self.mouse_pos[1] = event.mouse_region_y
        return self.execute(context)

    def execute(self, context):
        mode = str(context.mode)
        has_componentmodes = {"MESH", "ARMATURE", "GPENCIL"}
        has_editmode_only = {"CURVE", "SURFACE", "LATTICE", "META", "HAIR", "FONT"}

        # Mouse Over select option
        if context.preferences.addons[__name__].preferences.selmode_mouse and context.space_data.type == "VIEW_3D":
            og_obj = context.object

            if mode != "OBJECT":
                bpy.ops.object.mode_set(mode="OBJECT")

            bpy.ops.view3d.select(extend=False, deselect=False, toggle=False, deselect_all=False, center=False,
                                  enumerate=False, object=False, location=self.mouse_pos)

            sel_obj = context.object

            if og_obj == sel_obj:
                if mode == "OBJECT" and sel_obj.type in (has_componentmodes | has_editmode_only):
                    bpy.ops.object.editmode_toggle()
                return {"FINISHED"}

            if mode != "OBJECT" and sel_obj.type in (has_componentmodes | has_editmode_only):
                bpy.ops.object.editmode_toggle()
            else:
                mode = "OBJECT"

        # Set selection mode
        if context.active_object is not None:
            obj = context.active_object
        else:
            obj = context.object

        if obj.type in has_componentmodes:

            if self.edit_mode != "OBJECT":

                if obj.type == "ARMATURE":
                    bpy.ops.object.posemode_toggle()

                elif obj.type == "GPENCIL":
                    if mode == "OBJECT":
                        bpy.ops.gpencil.editmode_toggle()
                    if self.edit_mode == "VERT":
                        context.scene.tool_settings.gpencil_selectmode_edit = 'POINT'
                    elif self.edit_mode == "EDGE":
                        context.scene.tool_settings.gpencil_selectmode_edit = 'STROKE'
                    elif self.edit_mode == "FACE":
                        obj.data.use_curve_edit = not obj.data.use_curve_edit
                else:
                    if mode == "OBJECT" or mode == "SCULPT":
                        bpy.ops.object.editmode_toggle()
                    bpy.ops.mesh.select_mode(type=self.edit_mode)
            else:
                if obj.type == "GPENCIL":
                    bpy.ops.gpencil.editmode_toggle()
                else:
                    bpy.ops.object.editmode_toggle()

        elif obj.type in has_editmode_only:
            bpy.ops.object.editmode_toggle()

        else:
            return {"CANCELLED"}

        return {"FINISHED"}


class KedSpaceToggle(Operator):
    bl_idname = "ked.spacetoggle"
    bl_label = "Space Toggle"
    bl_description = "Space Toggle: Toggle between Edit and Object modes on selected object\n" \
                     "when mouse pointer is over -nothing-"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_options = {'REGISTER'}

    mouse_pos = Vector((0, 0))

    @classmethod
    def poll(cls, context):
        return (context.object is not None and
                context.object.type in {'MESH', 'CURVE', 'SURFACE', 'META', 'FONT', 'HAIR', 'GPENCIL'})

    def invoke(self, context, event):
        self.mouse_pos[0] = event.mouse_region_x
        self.mouse_pos[1] = event.mouse_region_y
        return self.execute(context)

    def execute(self, context):
        sel_mode = str(context.mode)
        bpy.ops.object.mode_set(mode='OBJECT')
        obj, hit_wloc, hit_normal, face_index = mouse_raycast(context, self.mouse_pos)

        if not face_index:
            if sel_mode == "OBJECT":
                bpy.ops.object.mode_set(mode="EDIT")
            elif sel_mode == "EDIT_MESH":
                bpy.ops.object.mode_set(mode="OBJECT")

        return {'FINISHED'}


class KedCallPie(Operator):
    # lazy custom pie call for ease of use with preset addon shortcuts to match idname
    bl_idname = "ked.call_pie"
    bl_label = "kedCallPie"

    name: StringProperty()

    def invoke(self, context, event):
        if context.space_data.type == 'VIEW_3D':
            bpy.ops.wm.call_menu_pie(name='%s' % self.name)
        return {'FINISHED'}


#
# PREFS
#
def update_panel(self, context):
    k = context.preferences.addons[__name__].preferences
    try:
        if "bl_rna" in UIModesModule.__dict__:
            bpy.utils.unregister_class(UIModesModule)
        UIModesModule.bl_category = k.category
        if k.load_modes:
            bpy.utils.register_class(UIModesModule)
    except Exception as e:
        print("keModes - Panel update failed :", e)
        pass


def add_extras(k):
    if k.ext_tools:
        bpy.types.VIEW3D_HT_tool_header.append(draw_tool_options)


def remove_extras():
    try:
        bpy.types.VIEW3D_HT_tool_header.remove(draw_tool_options)
    except Exception as e:
        print('keKit Extras Unregister Fail: ', e)
        pass


def reload_extras(self, context):
    k = bpy.context.preferences.addons[__name__].preferences
    remove_extras()
    add_extras(k)


def draw_kmi_row(kc, km, kmi, col):
    row = col.row()
    row.context_pointer_set("keymap", km)
    rna_keymap_ui.draw_kmi([], kc, km, kmi, row, 0)


def get_keymap_items(km):
    kmis = []
    for addon_km, addon_kmi in addon_keymaps:
        user_kmi = []
        # only using the one km category: only checking kmis
        for ukmi in km.keymap_items:
            if ukmi.idname == addon_kmi.idname:
                # Match via props (Can't just match idname: Same idname can use different props)
                if list(ukmi.properties.items()) == list(addon_kmi.properties.items()):
                    user_kmi = ukmi
        if user_kmi:
            kmis.append(user_kmi)
    return kmis


def set_keymap_item(km, idname, properties, state):
    # De-activated & set to (almost) unused key-combo to avoid shortcut conflicts
    kmi = km.keymap_items.new(idname=idname,
                              type='F20', value='PRESS', ctrl=True, alt=True, shift=True)
    if properties:
        for key, value in properties:
            setattr(kmi.properties, key, value)
    kmi.active = state
    addon_keymaps.append((km, kmi))


def draw_tool_options(self, context):
    k = context.preferences.addons[__name__].preferences
    if k.ext_tools:
        f = k.ext_factor
        t = context.tool_settings
        layout = self.layout
        row = layout.row(align=True)

        if context.mode == "OBJECT" and context.object:
            if context.object.type == "MESH":
                row.enabled = False
                row.separator(factor=1)
                row.label(icon='MOD_MIRROR')
                sub = row.row(align=True)
                sub.scale_x = 0.6
                sub.prop(context.object, "use_mesh_mirror_x", text="X", toggle=True)
                sub.prop(context.object, "use_mesh_mirror_y", text="Y", toggle=True)
                sub.prop(context.object, "use_mesh_mirror_z", text="Z", toggle=True)
                row.prop(t, "use_mesh_automerge", text="")
                row.separator(factor=1)
                row.prop(t, "use_transform_correct_face_attributes", text="", icon="MOD_UVPROJECT", toggle=True)
                row.prop(t, "use_edge_path_live_unwrap", text="", icon="UV", toggle=True)
            else:
                row.separator(factor=1.8)
                row.label(text="Affect Only")
                row.prop(t, "use_transform_data_origin", text="", icon="PIVOT_INDIVIDUAL", toggle=True)
                row.prop(t, "use_transform_pivot_point_align", text="", icon="ORIENTATION_LOCAL", toggle=True)
                row.prop(t, "use_transform_skip_children", text="", icon="CON_CHILDOF", toggle=True)
                row.separator(factor=1.8)

        elif context.mode == "EDIT_MESH":
            row.separator(factor=0.6)
            row.prop(t, "use_transform_correct_face_attributes", text="", icon="MOD_UVPROJECT", toggle=True)
            row.prop(t, "use_edge_path_live_unwrap", text="", icon="UV", toggle=True)

        elif context.mode == "EDIT_ARMATURE":
            # + disabled objmode settings to fill void
            row.enabled = False
            row.separator(factor=3)
            row.prop(t, "use_transform_data_origin", text="", icon="PIVOT_INDIVIDUAL", toggle=True)
            row.prop(t, "use_transform_pivot_point_align", text="", icon="ORIENTATION_LOCAL", toggle=True)
            row.prop(t, "use_transform_skip_children", text="", icon="CON_CHILDOF", toggle=True)
            row.separator(factor=3.8)

        elif context.mode == "PAINT_TEXTURE":
            row.separator(factor=11)

        elif context.mode == "PAINT_VERTEX":
            row.separator(factor=19.5)

        elif context.mode == "PAINT_WEIGHT":
            row.separator(factor=7.5)

        else:
            # meh .do nothing.
            pass

        layout.separator(factor=f)


class KedPrefs(AddonPreferences):
    bl_idname = __name__

    category: StringProperty(
            name="Tab Category",
            description="Choose a name (category) for tab placement",
            default="kekit",
            update=update_panel
            )
    selmode_mouse: BoolProperty(name="", description="Mouse-Over Element Mode Selection for Sel.mode script",
                                default=False)
    load_modes: BoolProperty(name="Modes", description="Enable/Disable 'Modes' panel",
                             default=True, update=update_panel)
    show_misc: BoolProperty(name="Misc", default=False, description="Show Misc Operators")
    ext_tools: BoolProperty(default=False, name="Extend Tool Settings", update=reload_extras,
                            description="Extend Tool Settings menu with additional buttons\n"
                                        "Also displays buttons (but disabled) in Object Mode (to show state)")
    ext_factor: FloatProperty(default=1.5, name="Separator factor", min=0, max=999,
                          description="UI-separator value the Extend Tool Settings menu is offset (from the right)")

    def draw(self, context):
        layout = self.layout
        row = layout.row(align=True)
        row.prop(self, "load_modes", text=" Modes Module")
        if self.load_modes:
            row.label(text="Modes Panel Location:")
            row.prop(self, "category", text="")

        row = layout.row()
        split = row.split(factor=0.2)
        split.label(text="Tool Settings:")
        row2 = split.row(align=True)
        row2.prop(self, "ext_tools", toggle=True)
        row2.prop(self, "ext_factor")

        row = layout.row(align=True)
        row.label(text="Misc Operators:")
        row.prop(self, "show_misc", text="Show Misc Operators in Modes Module")

        row = layout.row()
        row.label(text="Shortcuts:")
        col = layout.column()

        # 'Dynamic listing'
        wm = context.window_manager
        kc = wm.keyconfigs.user
        km = kc.keymaps.get('3D View Generic', None)
        col.label(text=str(km.name))

        kmis = get_keymap_items(km)
        if kmis:
            for kmi in kmis:
                draw_kmi_row(kc, km, kmi, col)

        row = layout.row()
        row.alignment = "CENTER"
        row.enabled = False
        row.label(text="To Reset: Disable & Re-enable Add-On")


#
# MODULE REGISTRATION
#
classes = (
    KedPrefs,
    KedSpaceToggle,
    KedSelectionMode,
    KedPieFitPrimAdd,
    KedPieOrientPivot,
    KedCallPie,
    KedModEmVis,
)

addon_keymaps = []

ked_pie_menus = [
    "VIEW3D_MT_ked_pie_fitprim_add", 
    "VIEW3D_MT_ked_pie_fit2grid", 
    "VIEW3D_MT_ked_pie_fit2grid_micro",
    "VIEW3D_MT_ked_pie_orientpivot",
    ]

ked_ops = [
    "ked.spacetoggle", 
    "ked.mod_em_vis"
    ]


def register():
    for c in classes:
        bpy.utils.register_class(c)

    k = bpy.context.preferences.addons[__name__].preferences
    if k.load_modes:
        try:
            if "bl_rna" in UIModesModule.__dict__:
                bpy.utils.unregister_class(UIModesModule)
            UIModesModule.bl_category = k.category
            bpy.utils.register_class(UIModesModule)
        except Exception as e:
            print("keModes - Panel update failed :", e)
            pass

    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon
    if kc:
        # km
        km = wm.keyconfigs.addon.keymaps.new(name='3D View Generic', space_type='VIEW_3D')
        # Selmode
        for i in ("VERT", "EDGE", "FACE", "OBJECT"):
            set_keymap_item(km, "ked.selmode", [("edit_mode", i)], False)
        # Spacetoggle
        for i in ked_ops:
            set_keymap_item(km, i, None, False)
        # Pie Menus
        for pm_name in ked_pie_menus:
            set_keymap_item(km, "ked.call_pie", [("name", pm_name)], False)
        add_extras(k)


def unregister():
    remove_extras()
    for c in reversed(classes):
        bpy.utils.unregister_class(c)
    if "bl_rna" in UIModesModule.__dict__:
        bpy.utils.unregister_class(UIModesModule)

    for km, kmi in addon_keymaps:
        try:
            km.keymap_items.remove(kmi)
        except Exception as e:
            print(e)
    addon_keymaps.clear()


if __name__ == "__main__":
    register()
